/*****************************************************************************

 @(#) $Id: mtp_route.h,v 0.7.4.1 2001/02/18 09:44:19 brian Exp $

 -----------------------------------------------------------------------------

 Copyright (C) 1997-2001  Brian Bidulock <bidulock@dallas.net>

 All Rights Reserved.

 This program is free software; you can redistribute it and/or modify it under
 the terms of the GNU General Public License as published by the Free Software
 Foundation; either version 2 of the License, or (at your option) any later
 version.

 This program is distributed in the hope that it will be useful, but WITHOUT
 ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 details.

 You should have received a copy of the GNU General Public License along with
 this program; if not, write to the Free Software Foundation, Inc., 675 Mass
 Ave, Cambridge, MA 02139, USA.

 -----------------------------------------------------------------------------

 Last Modified $Date: 2001/02/18 09:44:19 $ by $Author: brian $

 *****************************************************************************/

#ifndef __LINUX_MTP_ROUTE_H__
#define __LINUX_MTP_ROUTE_H__

/*
 *  Here's what I had for a routing heirarchy from include/linux/ss7.h...
 */

typedef struct sockaddr_ss7     sockaddr_ss7;
typedef struct ss7_addr         ss7_addr;
typedef struct ss7_link         ss7_link;       /* laziness... */
typedef struct ss7_linkset      ss7_linkset;
typedef struct ss7_route        ss7_route;
typedef struct ss7_routeset     ss7_routeset;
typedef struct ss7mtp_cb        ss7mtp_cb;

#define SS7_RS_MASK_MEMBER   0x00ffffff 
#define SS7_RS_MASK_CLUSTER  0x00ffff00 /* for ANSI */
#define SS7_RS_MASK_NETWORK  0x00ff0000 /* for ANSI */

#define SS7RTAB_SIZE 256

#define SS7_RST_LOCAL       0x01     /* destination is local     -- cb valid     */
#define SS7_RST_INTERNAL    0x02     /* destination is internal  -- cb valid */
#define SS7_RST_ADJACENT    0x04     /* destination is adjacent  -- route valid  */
#define SS7_RST_REMOTE      0x08     /* destination is remote    -- routes valid */
#define SS7_RST_LOCATION    0x0f     /* location mask */

#define SS7_RST_MEMBER      0x10     /* routeset is for a member  destination */
#define SS7_RST_CLUSTER     0x20     /* routeset if for a cluster destination */
#define SS7_RST_NETWORK     0x40     /* routeset if for a network destination */
#define SS7_RST_SCOPE       0x70     /* scope mask */

#define SS7_RST_ALIAS       0x80     /* destination is an alias */

#define SS7_RS_INHIBITED    0x01    /* the routeset is inhibited */
#define SS7_RS_PROHIBITED   0x02    /* the routeset is prohibited */
#define SS7_RS_RESTRICTED   0x04    /* the routeset is restricted */
#define SS7_RS_ALLOWED      0x08    /* the routeset is allowed */
#define SS7_RS_CONGESTED    0x10    /* we received TFC for this routeset */
#define SS7_RS_MTP_RESTART  0x20    /* point code in route is in MTP restart */
#define SS7_RS_TFPWAIT      0x40    /* bit set when in TFP wait state */
#define SS7_RS_TRA_SENT     0x80    /* TRA sent for restarting MTP */

#define SS7_RS_DONTUSE      ( SS7_RS_INHIBITED | SS7_RS_PROHIBITED | SS7_RS_MTP_RESTART )
                            /* dont use route if (flags & SS7_RS_DONTUSE) */

struct ss7_routeset
{
    __u32 addr;                     /* remote, local or local alias address */
    __u32 mask;                     /* mask for cluster routes */
    __u8 type;                      /* location and scope of entry */
    __u8 flags;                     /* availability */
    __u8 cong_status;               /* routeset remote congestion status */
    struct timer_list t;            /* timer for this destination */
    struct timer_list tmr;          /* MTP restart timer for this destination */
    struct sk_buff *skb;            /* buffer fot srsct (RCP) message */
    struct ss7mtp_cb* cb;           /* user parts for local point code */
    struct ss7_route *routes;       /* routes in this routeset */
    struct ss7_routeset *next;      /* list of other routesets in routinglist */
    struct ss7_routeset *collide;   /* hash collision list */
};

#define SS7_RT_INHIBITED        SS7_RS_INHIBITED   /* the route is inhibited */
#define SS7_RT_PROHIBITED       SS7_RS_PROHIBITED  /* we received TFP for this route */
#define SS7_RT_RESTRICTED       SS7_RS_RESTRICTED  /* we received TFR for this route */
#define SS7_RT_ALLOWED          SS7_RS_ALLOWED     /* we received TFA for this route */
#define SS7_RT_CONGESTED        SS7_RS_CONGESTED   /* signalling linkset is congested */
#define SS7_RT_MTP_RESTART      SS7_RS_MTP_RESTART /* adjacent point code in MTP restart */
#define SS7_RT_RSTEST           0x40  /* performing route set test */
#define SS7_RT_CONTROLLEDRR     0x80  /* performing controlled rerout on this route */

#define SS7_RT_DONTUSE          SS7_RS_DONTUSE     /* the route is unusable if(flags & SS7_RT_DONTUSE) */

struct ss7_route
{
    __u8 flags;                     /* flags for this route */
    __u8 last;                      /* last linkset used */
    __u8 priority;                  /* priority of route within routeset */
    struct timer_list t;            /* timer for this route */
    struct sk_buff *skb;            /* buffer for srst message */
    struct sk_buff_head ccrb;       /* controlled reroute buffer */
    struct ss7_linkset *linkset;    /* linksets for this route */
    struct ss7_routeset *routeset;  /* routeset that we belong to */
    struct ss7_route *other;        /* other route in a route pair */
    struct ss7_route *rset;         /* other routes in this routeset */
    struct ss7_route *lset;         /* other routes for this (combined)linkset */
    struct ss7_route *next;         /* list of all routes */
};

struct ss7_link;

#define SS7_LS_TYPE_A  2    /* according to fixed priority of links */
#define SS7_LS_TYPE_B  5
#define SS7_LS_TYPE_C  6
#define SS7_LS_TYPE_D  4
#define SS7_LS_TYPE_E  3
#define SS7_LS_TYPE_F  1

struct ss7_linkset
{
    __u8 type;                      /* combined linkset type */
    __u8 index;                     /* index of this linkset in combined linkset */
    __u16 flags;                    /* flags for this linkset */
    __u8 usable;                    /* number of usable links in linkset */
    struct ss7_routeset *local;     /* routeset entry for local point code */
    struct ss7_routeset *adjacent;  /* routeset entry for adjacent point code */
    struct ss7_link *links[16];     /* links in this linkset */
    struct ss7_route *routes;       /* route that we belong to */
    struct ss7_linkset *other;      /* other linkset in this combined linkset */
    struct ss7_linkset *next;       /* list of all linksets */
};

#define SS7_IFT_HARDWARE    1   /* hardware SS7 link */
#define SS7_IFT_MTP2UDP     2   /* UDP link emulation */
#define SS7_IFT_M2PEER      3   /* SIGTRAN M2PEER link emulation */
#define SS7_IFT_M2UA        4   /* SIGTRAN M2UA link emulation */

#define SS7_LINK_OUT_OF_SERVICE 0x0001 /* link is out of service */
#define SS7_LINK_INACTIVE       0x0002 /* link has not been activated */
#define SS7_LINK_LOCAL_INHIBIT  0x0004 /* link has been inhibited locally */
#define SS7_LINK_REMOTE_INHIBIT 0x0008 /* link has been inhibited remotely */
#define SS7_LINK_CONGESTED      0x0010 /* link is congested */
#define SS7_LINK_BLOCKED        0x0020 /* link is blocked (remote processor outage) */
#define SS7_LINK_CHANGEOVER     0x0040 /* performing changeover */
#define SS7_LINK_CHANGEBACK     0x0080 /* performing changeback */
#define SS7_LINK_TEST           0x0100 /* first signalling link test */
#define SS7_LINK_RETEST         0x0200 /* on second SLTM test */

#define SS7_LINK_DONTUSE        0x000f /* the link is unusable if(flags & SS7_LINK_DONTUSE) */

#define SS7_LK_COB_IDLE            0    /* no changeover/changeback */
#define SS7_LK_COB_BSNT_REQ_COO    1    /* BSNT has been requested for order */
#define SS7_LK_COB_BSNT_REQ_COA    2    /* BSNT has been requested for ack */
#define SS7_LK_COB_TIME_CONTROLLED 3    /* Time controlled changeover/changeback in progress */
#define SS7_LK_COB_WAITING_ACK     4    /* COO/ECO or CBD sent, awaiting ACK */
#define SS7_LK_COB_RETRIEVING      5    /* Retreiving messages from link */

struct ss7_link
{
    __u8 type;                           /* type of device */
    __u8 index;                          /* index of this link in combined linkset (sls) */
    __u8 flags;                          /* flags for this link */
    __u8 cong_status;                    /* link congestion level */
    __u8 cob_state;                      /* changeover state */
    __u8 loc_bsnt;                       /* retrieved bsnt local */
    __u8 rem_bsnt;                       /* retrieved bsnt remote */
    __u8 reason;                         /* changeover cancel changeback reason */
    struct device *dev;                  /* device pointer for this link (real or virtual) */
    struct sk_buff *skb;                 /* message holder for changeover order */
    struct ss7_link* alt;                /* alternate link or source of changeover order */
    struct timer_list t;                 /* co/cb timer for this link */
    struct timer_list t1t;               /* t1 (SLTC ACK) timer for this link */
    struct timer_list t2t;               /* t2 (SLTC SND) timer for this link */
    struct sk_buff_head cocbb;           /* changeover/changeback buffer */
    struct sk_buff_head rtrvb;           /* retrieval buffer */
    struct ss7_linkset *linkset;         /* the linkset this link belongs to */
    struct ss7_link *next;               /* list of all links */
    void (*output)(struct sk_buff *skb); /* output method normally dev_queue_xmit(skb) */
    void (*input)(struct sk_buff *skb);  /* input method normally netif_rx(skb) */
};

/* from net/ss7/mtp_route.c */
extern struct ss7_routeset* ss7_all_routesets;
extern struct ss7_route*    ss7_all_routes;
extern struct ss7_linkset*  ss7_all_linksets;

extern unsigned             ss7_addr_type      (struct ss7_addr);
extern struct ss7_routeset* ss7_rt             (struct ss7_addr);
extern struct ss7_routeset* ss7_rs_add         (struct ss7_addr, __u8 flags, __u8 type);
extern struct ss7_routeset* ss7_rt_type        (struct ss7_addr, __u8 flags, __u8 type);

extern struct ss7_route*    ss7_add_route      (struct ss7_routeset *rem,
                                                struct ss7_routeset *loc,
                                                struct ss7_routeset *adj);
extern struct ss7_route*    ss7_select_rt      (struct ss7_routeset *rs);

extern struct ss7_link*     ss7_select_link_ls (struct ss7_route *rt);
extern struct ss7_link*     ss7_select_link    (struct ss7_route *rt, unsigned char sls);

extern int                  ss7_move_link      (struct ss7_link *iface,
                                                struct ss7_routeset *local,
                                                struct ss7_routeset *adjacent,
                                                unsigned char sls);
extern __u32 ss7_get_adjacent(struct sk_buff *skb);
extern void ss7_rtab_init(void);
extern void ss7_rtab_free(void);
int ss7_route_ioctl(unsigned int cmd, void *arg);

#endif  __LINUX_MTP_ROUTE_H__

