/*****************************************************************************

 @(#) $Id: sls_lk_sm.h,v 0.7.4.1 2001/02/18 09:44:36 brian Exp $

 -----------------------------------------------------------------------------

 Copyright (C) 1997-2001  Brian Bidulock <bidulock@dallas.net>

 All Rights Reserved.

 This program is free software; you can redistribute it and/or modify it under
 the terms of the GNU General Public License as published by the Free Software
 Foundation; either version 2 of the License, or (at your option) any later
 version.

 This program is distributed in the hope that it will be useful, but WITHOUT
 ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 details.

 You should have received a copy of the GNU General Public License along with
 this program; if not, write to the Free Software Foundation, Inc., 675 Mass
 Ave, Cambridge, MA 02139, USA.

 -----------------------------------------------------------------------------

 Last Modified $Date: 2001/02/18 09:44:36 $ by $Author: brian $

 *****************************************************************************/

#ifndef __LK_SM_H__
#define __LK_SM_H__

/*
 *  These state machines describe the Signalling Link information and behavior.
 *  They interact with the Level 2 state machines and the Signalling Link Set
 *  state machines only.
 */


#define LK_FLAG_OUT_OF_SERVICE  0x00000001  /* link is out of service at level 2            */
#define LK_FLAG_INACTIVE        0x00000002  /* link has not been activated                  */
#define LK_FLAG_LOC_INHIBITED   0x00000004  /* link is locally  inhibited                   */
#define LK_FLAG_REM_INHIBITED   0x00000008  /* link is remotely inhibited                   */
#define LK_FLAG_LOC_BLOCKED     0x00000010  /* link is locally  blocked (lpo)               */
#define LK_FLAG_REM_BLOCKED     0x00000020  /* link is remotely blocked (rpo)               */
#define LK_FLAG_ADJ_SP_RESTART  0x00000040  /* the adjacent SP is in restart                */
#define LK_FLAG_LOC_SP_RESTART  0x00000080  /* the local    SP is in restart                */
#define LK_FLAG_INACCESSIBLE    0x00000100  /* adjacent SP is inaccessible                  */
#define LK_FLAG_CRITICAL        0x00000200  /* critical (last available) link in linkset    */
#define LK_FLAG_ROUTES_LOADED   0x00000400  /* the link involved in loaded   routes         */
#define LK_FLAG_ROUTES_CRITICAL 0x00000800  /* the link involved in critical routes         */
#define LK_FLAG_1ST_SLT         0x00001000  /* 1st signalling link test in progress         */
#define LK_FLAG_2ND_SLT         0x00002000  /* 2nd signalling link test in progress         */
#define LK_FLAG_CHANGEOVER      0x00004000  /* changeover in progress                       */
#define LK_FLAG_CHANGEBACK      0x00008000  /* changeback in progress                       */
#define LK_FLAG_BSNT_REQ_COO    0x00010000  /* BSNT has been requested for COO              */
#define LK_FLAG_BSNT_REQ_COA    0x00020000  /* BSNT has been requested for COA              */
#define LK_FLAG_WAITING_CO_ACK  0x00040000  /* Awaiting changeover ack                      */
#define LK_FLAG_WAITING_CB_ACK  0x00080000  /* Awaiting changeback ack                      */
#define LK_FLAG_WAITING_CB_ACK2 0x00100000  /* Awaiting changeback ack (2nd attempt)        */
#define LK_FLAG_RETRIEVING      0x00200000  /* Retrieving buffers                           */
#define LK_FLAG_FSNC_PROVIDED   0x00400000  /* FSNC was provided by other end               */
#define LK_FLAG_BSNT_ACQUIRED   0x00800000  /* BSNT was acquired by this  end               */
#define LK_FLAG_TIME_CTRL_CO    0x01000000  /* Time controlled changeover in progress       */
#define LK_FLAG_TIME_CTRL_CB    0x02000000  /* Time controlled changeback in progress       */
#define LK_FLAG_INHIBITING      0x04000000  /* local inhibiting in progress                 */
#define LK_FLAG_INHIBITING2     0x08000000  /* local inhibiting in progress (2nd attempt)   */ 
#define LK_FLAG_UNINHIBITING    0x10000000  /* uninhibiting in progress                     */
#define LK_FLAG_UNINHIBITING2   0x20000000  /* uninhibiting in progress (2nd attempt)       */
#define LK_FLAG_FORCEUNINHIBIT  0x40000000  /* force uninhibiting in progress               */
#define LK_FLAG_FORCEUNINHIBIT2 0x80000000  /* force uninhibiting in progress (2nd attempt) */

#define LK_MASK_UNUSABLE        0x000000f3  /* mask for checking usability for mgmt msgs    */
#define LK_MASK_UNAVAILABLE     0x000000ff  /* mask for checking availability of link       */
#define LK_MASK_NETWORK_CONFIG  0x00000fc0  /* mask for network configuration flags         */

enum {
    LK_REASON_UNSPECIFIED = 0,      /* unspecified (unknown)                */
    LK_REASON_INHIBITED,            /* the link is inhibited                */
    LK_REASON_UNINHIBITED,          /* the link is uninhibited              */
    LK_REASON_INHIBITING,           /* inhibiting in progress               */
    LK_REASON_UNINHIBITING,         /* uninhibiting in progress             */
    LK_REASON_BUSY,                 /* the action is already in progress    */
    LK_REASON_INACCESSIBLE,         /* the remote SP is inaccessible        */
    LK_REASON_CRITICAL,             /* the link is critical                 */
    LK_REASON_MAXIMUM_RETRIES       /* maximum number of retries attempts   */
};

static inline void
lk_network_configuration_change(lk_t *lk, unsigned long flags) /* from LS */
{
    unsigned long newflags, chgflags;
    newflags = chgflags = lk->statem.flags;
    /*
     *  Rather than requesting the information as to how to perform routing at
     *  the time that management messages come in (as used in the Q.704 SDLs)
     *  this approach has the routing control update the link set and link
     *  routing information when it changes.
     *
     *  This works on the principle that routing changes are few and far apart
     *  (because it takes management action or failure to create a routing
     *  change) whereas the access to the data is frequent (with every message
     *  which is routed).  Therefore, cacheing the information local to the
     *  decision is the best approach.
     */
    newflags &= ~LK_MASK_NETWORK_CONFIG;
    newflags |= (flags & LK_MASK_NETWORK_CONFIG);
    lk->statem.flags = newflags;
    chgflags ^= newflags;
    if ( chgflags & LK_FLAG_LOC_SP_RESTART ) {
        if ( newflags & LK_FLAG_LOC_SP_RESTART ) {
            /*
             *  Perform OWN RESTART BEGINS actions here.
             */
        } else {
            /*
             *  Perform OWN RESTART ENDS actions here.
             */
        }
    }
    if ( chgflags & LK_FLAG_ADJ_SP_RESTART ) {
        if ( newflags & LK_FLAG_ADJ_SP_RESTART ) {
            /*
             *  Perform ADJ SP RESTARTING actions here.
             */
        } else {
            /*
             *  Perform ADJ SP RESTARTED actions here.
             */
        }
    }
    lk->statem.flags = newflags;
}
/*
 *  10      Management Inhibiting
 *
 *  10.1    General
 *
 *  Signalling link management inhibiting is requested by management when it
 *  becomes necessary, e.g. for maintenance or testing purposes (for example, if
 *  the link experiences too many changeovers and changebacks in a short time,
 *  or there is a significant link error rate), to make or keep a signalling
 *  link unavailable to User Part-generated signalling traffic.  Management
 *  inhibiting is a signalling traffic management action, and does not cause any
 *  link status changes at level 2.  A signalling link is marked "inhibited"
 *  under the management inhibiting procedure.  In particular, a signalling link
 *  that was active and in service prior to being inhibited will remain so, and
 *  will thus be able to transmit maintenance and test messages, for which the
 *  signalling link test message could be used (see clause 2/Q.707).
 *
 *  Inhibiting of a signalling link may be requested by management functions at
 *  either end of the link.  The request is granted, provided that the
 *  inhibiting action does not cause any previously accessible destination to
 *  become inaccessible at either end of the signalling link.  The request may
 *  also be refused under certain circumstances such as congestion.
 *
 *  A signalling link normally remains inhibited until uninhibiting is invoked
 *  in the signalling point at which inhibiting was initiated. Uninhibiting is
 *  initiated either at the request of a management function or by routing
 *  functions at either end of the signalling link when it is found that a
 *  destination has become inacessible for signalling traffic and the link sets
 *  associated with routes to that destination contains inhibited links.  Unless
 *  unavailable for other reasons, unihibiting causes the signalling link to
 *  enter the available state and changeback to be initiated.
 *
 *  Periodic tests are made on the inhibit status of inhibited links.  Such
 *  periodic tests should not add significantly to the traffic load on the
 *  signalling network, and remove the need for a signalling point to perform
 *  inhibit tests at signalling point restart.
 *
 *  If a test on the inhibit status of a link reveals discrepancies between the
 *  signalling points at each end of the link, the link is either uninhibited or
 *  force uninhibited as appropriate, to align the inhibit status at each end of
 *  the link.
 *
 *  10.2    Inhibiting initiation and actions
 *
 *  When at signalling point X a request is received from a management function
 *  to inhibit a signalling link to signalling point Y, the following action
 *  take place:
 *
 *  a)  A check if performed at signalling point X to determine whether, in the
 *      case of an available link, inhibiting will result in a destination
 *      becoming inaccessible, or in the case of an unavailable link, signalling
 *      point Y is inaccesible.  If either is the case, management is informed
 *      that the inhibiting request is denied.
 *
 *  b)  If inhibiting is permitted, signalling point X sends an inhibit message
 *      to signalling point Y indicating that it wishes to inhibit the
 *      signalling link identified in the message.
 *
 *  c)  Signalling point Y, on receiving the inhibit message from X, checks
 *      whether, in the case of an available link, inhibiting will result in a
 *      destination becoming inaccessible and, if so, an inhibit denied message
 *      is returned to signalling point X.  The latter then informs the
 *      management function which requested inhibiting that the request cannot
 *      be granted.
 *
 *  d)  If the signalling point Y finds that inhibiting of the concerned link is
 *      permissible, it sends an inhibit acknowledgement to signalling point X
 *      and marks the link remotely inhibited.
 *
 *      If the link concerned is currently carrying traffic, signalling point Y
 *      sends the inhibit acknowledgement via that link and diverts subsequent
 *      traffic for it, using the time controlled changeover procedure.  Y then
 *      starts inhibit test timer T23.
 *
 *  e)  On receiving an inhibit acknowledgement message, signalling point X
 *      marks the link locally inhibited and informs management that the link is
 *      inhibited.
 *
 *      If the link concerned is currently carrying traffic, signalling point X
 *      diverts subsequent traffic for that link, using the time-controlled
 *      changeover procedure.  X then starts inhibit test timer T22.
 *
 *  f)  When changeover has been completed, the link while inhibited, will be
 *      unavailable for the transfer of user-generated traffic but still permits
 *      the exchange of test messages.
 *
 *  g)  If, for any reason, the inhibit acknowledgement message is not received,
 *      a timer T14 expires and the procedure is restarted including inspection
 *      of the status of the destination of the inhibit message.  If the
 *      destination is not available, management is informed.
 *
 *  At most two consecutive automatic attempts may be made to inhibit a
 *  particular signalling link.
 *
 *  A signalling point may not transmit an inhibit message for a particular
 *  signalling link if it has already transmitted an uninhibit message for that
 *  link, and neither an acknowledgement for that uninhibit message has been
 *  received nor has the uninhibit procedure finally timed out.
 */

static inline void
lk_management_inhibit(lk_t *lk) /* from MGMT */
{
    register unsigned long *f = lk->statem.flags;
    /*
     *  Q.704/10.2 a): A check if performed at signalling
     *  point X to determine whether, in the case of an
     *  available link, inhibiting will result in a
     *  destination becoming inaccessible, or in the case of
     *  an unavailable link, signalling point Y is
     *  inaccesible.  If either is the case, management is
     *  informed that the inhibiting request is denied.
     */
    if ( *f & LK_FLAG_LOC_INHIBITED ) {
        lk_management_inhibit_denied(lk, LK_REASON_INHIBITED);
        return;
    }
    if ( *f & LK_FLAG_INHIBITING ) {
        lk_management_inhibit_denied(lk, LK_REASON_BUSY);
        return;
    }
    if ( *f & LK_FLAG_UNINHIBITING ) {
        /*
         *  Q.704/10.2: ... A signalling point may not
         *  transmit an inhibit message for a particular
         *  signalling link if it has already transmitted an
         *  uninhibit message for that link, and neither an
         *  acknowledgement for that uninhibit message has
         *  been received nor has the uninhibit procedure
         *  finally timed out.
         */
        lk_management_inhibit_denied(lk, LK_REASON_UNINHIBITING);
        return;
    }
    if ( *f & LK_FLAG_INACCESSIBLE ) {
        lk_management_inhibit_denied(lk, LK_REASON_INACCESSIBLE);
        return;
    }
    if ( (*f & LK_FLAG_ROUTES_CRITICAL) && (*f & LK_FLAG_CRITICAL) ) {
        lk_management_inhibit_denied(lk, LK_REASON_CRITICAL);
        return;
    }
    /*  
     *  Q.704/10.2 b): If inhibiting is permitted,
     *  signalling point X sends an inhibit message to
     *  signalling point Y indicating that it wishes to
     *  inhibit the signalling link identified in the
     *  message.
     */
    *f |= LK_FLAG_INHIBITING;
    lk_timer_start(t14);
    lk_send_inhibit(lk);
}

static void
lk_t14_timeout(lk_t *lk) /* T14 timeout */
{
    register unsigned long *f = &lk->statem.flags;
    /*
     *  Q.704/10.2 g): If, for any reason, the inhibit
     *  acknowledgement message is not received, a timer T14
     *  expires and the procedure is restarted including
     *  inspection of the status of the destination of the
     *  inhibit message.  If the destination is not
     *  available, management is informed.
     */
    if ( !(*f & LK_FLAG_INHIBITING) ) return;
    *f &= ~LK_FLAG_INHIBITING;
    if ( *f & LK_FLAG_INHIBITING2 ) {
        *f &= ~LK_FLAG_INHIBITING2;
        lk_management_inhibit_denied(lk, LK_REASON_MAXIMUM_RETRIES);
    }
    else
    {
        if ( *f & LK_FLAG_INACCESSIBLE ) {
            lk_management_inhibit_denied(lk, LK_REASON_INACCESSIBLE);
            return;
        }
        if ( (*f & LK_FLAG_ROUTES_CRITICAL) && (*f & LK_FLAG_CRITICAL) ) {
            lk_management_inhibit_denied(lk, LK_REASON_CRITICAL);
            return;
        }
        *f |= LK_FLAG_INHIBITING | LK_FLAG_INHIBITING2;
        lk_timer_start(t14);
        lk_send_inhibit(lk);
    }
}

static inline void
lk_remote_inhibit(lk_t *lk) /* LIN from adjacent L3 peer */
{
    register unsigned long *f = &lk->statem.flags;
    /*
     *  Q.704/10.2 c): Signalling point Y, on receiving the
     *  inhibit message from X, checks whether, in the case
     *  of an available link, inhibiting will result in a
     *  destination becoming inaccessible and, if so, an
     *  inhibit denied message is returned to signalling
     *  point X.  The latter then informs the management
     *  function which requested inhibiting that the request
     *  cannot be granted.
     */
    if ( *f & (LK_FLAG_LOC_INHIBITED | LK_FLAG_REM_INHIBITED) ) {
        /*
         *  Q.704/10.4 a) -  An inhibit signalling link
         *  message concerning an inhibited signalling link
         *  is answered with an inhibit acknowledgement
         *  message without taking any further action.
         */
        lk_send_inhibit_ack(lk);
        return;
    }
    if ( *f & LK_FLAG_INACCESSIBLE ) {
        /* nothing to do: lost communication with adjacent SP */
        return;
    }
    if ( (*f & LK_FLAG_ROUTES_CRITICAL) && (*f & LK_FLAG_CRITICAL) ) {
        lk_send_inhibit_denied(lk);
        return;
    }
    /*
     *  Q.704/10.2 d): If the signalling point Y finds that
     *  inhibiting of the concerned link is permissible, it
     *  sends an inhibit acknowledgement to signalling point
     *  X and marks the link remotely inhibited.
     */
    lk_send_inhibit_ack(lk);    /* must be on concerned link if possible */
    *f |= LK_FLAG_REM_INHIBITED;
    /*
     *  Q.704/10.2 d): ... If the link concerned is
     *  currently carrying traffic, signalling point Y sends
     *  the inhibit acknowledgement via that link and
     *  diverts subsequent traffic for it, using the time
     *  controlled changeover procedure.  Y then starts
     *  inhibit test timer T23.
     */
    if ( !(*f & (LK_MASK_UNAVAILABLE & ~LK_FLAG_REM_INHIBITED)) )
        if ( *f & LK_FLAG_ROUTES_LOADED )
            lk_time_controlled_changeover(lk);
    lk_timer_start(t23);
}

static inline void
lk_inhibit_ack(lk_t *lk) /* LIA from adjacent L3 peer */
{
    register unsigned long *f = &lk->statem.flags;

    if ( !(*f & LK_FLAG_INHIBITING ) ) {
        /*
         *  Q.704/10.4 d): If an inhibit acknowledgement
         *  message is received and no inhibit signalling
         *  link message is outstanding for the concerned
         *  link, no action is taken.
         */
        return;
    }
    lk_timer_stop(t14);
    *f &= ~(LK_FLAG_INHIBITING | LK_FLAG_INHIBITING2);
    /*
     *  Q.704/10.2 e): On receiving an inhibit
     *  acknowledgement message, signalling point X marks
     *  the link locally inhibited and informs management
     *  that the link is inhibited.
     */
    *f |= LK_FLAG_LOC_INHIBITED;
    lk_management_inhibited(lk);
    /*
     *  Q.704/10.2 e): ... If the link concerned is
     *  currently carrying traffic, signalling point X
     *  diverts subsequent traffic for that link, using the
     *  time-controlled changeover procedure.  X then starts
     *  inhibit test timer T22.
     */
    if ( !(*f & (LK_MASK_UNAVAILABLE & ~LK_FLAG_LOC_INHIBITED)) )
        if ( *f & LK_FLAG_ROUTES_LOADED )
            lk_time_controlled_changeover(lk);
    lk_timer_start(t22);
}
/*
 *  10.3    Uninhibited initiation and actions
 *
 *  Signalling link uninhibiting is initiated at the signalling point which
 *  originally caused the link to be inhibited, upon receipt of an uninhibit or
 *  forced uninhibit request.
 *
 *  In a given signalling point, an uninhibit request may be initiated for a
 *  locally inhibited link by the management or signalling route control
 *  function, while a forced uninhibit request may be initiated for a remotely
 *  inhibited link by the signalling routing control function only.
 *
 *  Signalling routing control will initiate signalling link uninhibit if an
 *  inhibited link is found to be a member of a link set in a route to a
 *  destination which has become inaccessible.
 *
 *  If such signalling routing control uninhibiting were unsuccessful because of
 *  a failed or block inhibited link, and if that link later recovers or becomes
 *  unblocked with the destination still unavailable, uninhibiting is
 *  re-attempted.
 *
 *  A signalling point may not transmit an uninhibit message for a particular
 *  signalling link if it has already transmitted an inhibit message for that
 *  link, and neither an acknowledgement for that inhibit message has been
 *  received nor has the inhibit procedure finally timed out.
 *
 *  10.3.1  Management-initiated uninhibiting
 *
 *  Upon receipt of an uninhibiting request from the management function of
 *  signalling point X regarding an inhibited link to signalling point Y, the
 *  following actions take place:
 *
 *  a)  A check is performed at signlaling point X to determine whether an
 *      uninhibit message can be sent to signalling point Y, either over an
 *      available route, or if all routes to signalling point Y are unavailable,
 *      over the concerned inhibited link.  If all routes to signalling point Y
 *      are unavailable and the concerned inhibited link is marked failed or
 *      processor outage, management is informed that uninhibiting is not
 *      possible.
 *
 *  b)  If uninhibiting is possible, signalling point X sends uninhibit
 *      signalling link message to signalling point Y indicating that the link
 *      identified in the message should be uninhibited.
 *
 *  c)  Upon receipt of the uninhibit link message, signalling point Y returns
 *      an uninhibit acknowledgement message to signalling point X and cancels
 *      the remote inhibit indication.  If no local inhibited, failed or blocked
 *      condition exists on the link, it is put in the available state and
 *      changeback is initiated.
 *
 *  d)  On receipt of the uninhibit acknowledgement message, signalling point X
 *      cancels the local inhibit indication and informs management that the
 *      link has be uninhibited.  If no remote inhibited, failed or blocked
 *      condition exists on the link, it is put in the available state and
 *      changeback is initiated.
 *
 *  e)  If, for any reason, the uninhibit acknowledgement message is not
 *      received, a timer T12 expires.  If this is the first expirty of T12 for
 *      this uninhibition attempt on this link, the procedure is restarted
 *      including inspection of the status of the destination of the uninhibit
 *      message.  If the destination is not available, or T12 has expired for
 *      the second time during the uninhibition attempt on this link, management
 *      is informed, and the uninhibition is abandoned.
 */

static inline void
lk_management_uninhibit(lk_t *lk) /* from MGMT */
{
    unsigned long *f = &lk->statem.flags;
    /*
     *  Q.704/10.3.1 a): A check is performed at signalling
     *  point X to determine whether an uninhibit message
     *  can be sent to signalling point Y, either over an
     *  available route, or if all routes to signalling
     *  point Y are unavailable, over the concerned
     *  inhibited link.  If all routes to signalling point Y
     *  are unavailable and the concerned inhibited link is
     *  marked failed or processor outage, management is
     *  informed that uninhibiting is not possible.
     */
    if ( !(*f & (LK_FLAG_LOC_INHIBITED | LK_FLAG_REM_INHIBITED)) ) {
        lk_management_uninhibit_denied(lk, LK_REASON_UNINHIBITED);
        return;
    }
    if ( *f & LK_FLAG_UNINHIBITING ) {
        lk_management_uninhibit_denied(lk, LK_REASON_BUSY);
        return;
    }
    if ( *f & LK_FLAG_INHIBITING ) {
        /*
         *  Q.704/10.3: ... A signalling point may not
         *  transmit an uninhibit message for a particular
         *  signalling link if it has already transmitted an
         *  inhibit message for that link, and neither an
         *  acknowledgement for that inhibit message has
         *  been received nor has the inhibit procedure
         *  finally timed out.
         */
        lk_management_uninhibit_denied(lk, LK_REASON_INHIBITING);
        return;
    }
    if ( (*f & LK_FLAG_INACCESSIBLE) && (*f & LK_MASK_UNUSABLE) )
        lk_management_uninhibit_denied(lk, LK_REASON_INACCESSIBLE);
        return;
    }
    /*
     *  Q.704/10.3.1 b): If uninhibiting is possible,
     *  signalling point X sends uninhibit signalling link
     *  message to signalling point Y indicating that the
     *  link identified in the message should be
     *  uninhibited.
     */
    *f |= LK_FLAG_UNINHIBITING;
    lk_send_uninhibit(lk);
    lk_timer_start(t12);
}

static void
lk_t12_timeout(lk_t *lk) /* T12 timeout */
{
    register unsigned long *f = &lk->statem.flags;
    /*
     *  Q.704/10.3.1 e): If, for any reason, the uninhibit
     *  acknowledgement message is not received, a timer T12
     *  expires.  If this is the first expirty of T12 for
     *  this uninhibition attempt on this link, the
     *  procedure is restarted including inspection of the
     *  status of the destination of the uninhibit message.
     *  If the destination is not available, or T12 has
     *  expired for the second time during the uninhibition
     *  attempt on this link, management is informed, and
     *  the uninhibition is abandoned.
     */
    if ( !(*f & LK_FLAG_UNINHIBITING) ) return;
    *f &= ~LK_FLAG_UNINHIBITING;
    if ( *f & LK_FLAG_UNINHIBITING2 ) {
        *f &= ~LK_FLAG_UNINHIBITING2;
        if ( *f & LK_FLAG_FORCEUNINHIBIT ) {
            *f &= ~LK_FLAG_FORCEUNINHIBIT;
            lk_route_uninhibit_abandoned(lk);
        } else
            lk_management_uninhibit_denied(lk, LK_REASON_MAXIMUM_RETRIES);
    }
    else
    {
        if ( (*f & LK_FLAG_INACCESSIBLE) && (*f & LK_MASK_UNUSABLE) )
            lk_management_uninhibit_denied(lk, LK_REASON_INACCESSIBLE);
            return;
        }
        *f |= LK_FLAG_UNINHIBITING | LK_FLAG_UNINHIBITING2;
        lk_send_uninhibit(lk);
        lk_timer_start(t12);
    }
}

static inline void
lk_uninhibit(lk_t *lk) /* LUN from adjacent L3 peer */
{
    register unsigned long *f = &lk->statem.flags;
    if ( !(*f & (LK_FLAG_LOC_INHIBITED | LK_FLAG_REM_INHIBITED)) ) {
        /*
         * Q.704/10.4 b): An uninhibit signalling link
         * message concerning an uninhibited signalling link
         * is answered with an uninhibit acknowledgement
         * message without taking any further action.
         */
        lk_send_uninhibit_ack(lk);
        return;
    }
    if ( (*f & LK_FLAG_INACCESSIBLE) && (*f & LK_MASK_UNUSABLE) ) {
        /* nothing to do: communication lost with adjacent SP */
        return;
    }
    if ( *f & LK_FLAG_FORCEUNINHIBIT ) {
        lk_timer_stop(t13);
        *f &= ~(LK_FLAG_FORCEUNINHIBIT | LK_FLAG_FORCEUNINHIBIT2);
    }
    /*
     *  Q.704/10.3.1 c): Upon receipt of the uninhibit link
     *  message, signalling point Y returns an uninhibit
     *  acknowledgement message to signalling point X and
     *  cancels the remote inhibit indication.  If no local
     *  inhibited, failed or blocked condition exists on the
     *  link, it is put in the available state and
     *  changeback is initiated.
     */
    lk_send_uninhibit_ack(lk);  /* on inhibited link if necessary */
    *f &= ~LK_FLAG_REM_INHIBITED;
    if ( !(*f & LK_MASK_UNAVAILABLE) )
        lk_changeback(lk);
}

static inline void
lk_uninhibit_ack(lk_t *lk) /* LUA from adjacent L3 peer */
{
    register unsigned long *f = &lk->statem.flags;

    if ( !(*f & LK_FLAG_UNINHIBITING ) ) {
        /*
         *  Q.704/10.3.1 e)  If an uninhibit acknowledgement
         *  message is received and no unihibit signalling
         *  link message is outstanding for the concerned
         *  link, no action is taken.
         */
        return;
    }
    lk_timer_stop(t12);
    *f &= ~(LK_FLAG_UNINHIBITING | LK_FLAG_UNINHIBITING2);
    /*
     *  Q.704/10.3.1 d)  On receipt of the uninhibit
     *  acknowledgement message, signalling point X cancels
     *  the local inhibit indication and informs management
     *  that the link has be uninhibited.  If no remote
     *  inhibited, failed or blocked condition exists on the
     *  link, it is put in the available state and
     *  changeback is initiated.
     */
    *f &= ~LK_FLAG_LOC_INHIBITED;
    lk_management_uninhibited(lk);
    if ( !(*f & LK_MASK_UNAVAILABLE) )
        lk_changeback(lk);
}

/*
 *  10.3.2  Signalling routing control initiated uninhibiting
 *
 *  Upon receipt of an uninhibit request for signalling routing control at
 *  signalling point X regarding an inhibited link to signalling point Y, the
 *  following actions take place:
 *
 *  a)  A check is performed at signalling point X to determine whether the
 *      concerned inhibited link is marked failed or blocked.  If it is, then
 *      signalling point X is unable to transmit an uninhibit message to
 *      signalling point Y, uninhibiting is therefore not possible, and the
 *      uninhibiting attempt is abandoned.
 *
 *  b)  If unihibiting is possible, a further check is performed by signalling
 *      point X to determine whether inhibiting initiated by X (local
 *      inhibiting) or inhibiting initiated by Y (remote inhibiting) is in
 *      effect.
 *
 *  c)  If local inhibiting is in effect, then the actions described 10.3.1 b),
 *      c), d) and e) take place.  If uninhibition is abandoned, step f) below is
 *  taken.
 *
 *  d)  If remote inhibiting is in effect, the signalling point X requests
 *      forced uninhibiting of the signalling link by sending a force uninhibit
 *      signalling link message to signalling point Y, which will then initiate
 *      uninhibiting in accordance with the description given in 10.3.1 b), c),
 *      d) and e).
 *
 *      The force uninhibit signalling link message is transmitted down the link
 *      to be uninhibited.
 *
 *  e)  If, for any reason, an unihibit signalling link message is not received
 *      in response to the force uninhibit message, a timer T13 expires.  If
 *      this is the first expiry of T13 for this uninhibition attempt on this
 *      link, the procedure is restarted including inspection of the status of
 *      the inhibited link.  If the link is marked failed or blocked, or timer
 *      T13 has expired for the second time during inhibition of this link,
 *      management is informed and the uninhibition is abandoned.
 *
 *  f)  If an attempt to uninhibit a signalling link is abandoned, signalling
 *      routing control attempts to uninhibit the next inhibited link to
 *      signaling point Y, starting from a) above.  The search continues until
 *      either a link is successfully uninhibited or all possible links to Y in
 *      the routing table have been exhausted, or the destination has become
 *      available for other reasons.
 */

static inline void
lk_route_uninhibit(lk_t *lk) /* from LS */
{
    register unsigned long *f = &lk->statem.flags;

    if ( *f & LK_MARK_UNUSABLE ) {
        /*
         *  Q.704/10.3.2 a): A check is performed at
         *  signalling point X to determine whether the
         *  concerned inhibited link is marked failed or
         *  blocked.  If it is, then signalling point X is
         *  unable to transmit an uninhibit message to
         *  signalling point Y, uninhibiting is therefore
         *  not possible, and the uninhibiting attempt is
         *  abandoned.
         */
        /*
         *  Q.704/10.3.2 f): If an attempt to uninhibit a
         *  signalling link is abandoned, signalling routing
         *  control attempts to uninhibit the next inhibited
         *  link to signaling point Y, starting from a)
         *  above.  The search continues until either a link
         *  is successfully uninhibited or all possible
         *  links to Y in the routing table have been
         *  exhausted, or the destination has become
         *  available for other reasons.
         */
        lk_route_uninhibit_abandoned(lk);
        return;
    }
    /*
     *  Q.704/10.3.2 b): If unihibiting is possible, a
     *  further check is performed by signalling point X to
     *  determine whether inhibiting initiated by X (local
     *  inhibiting) or inhibiting initiated by Y (remote
     *  inhibiting) is in effect.
     */
    if ( *f & LK_FLAG_LOC_INHIBITED ) {
        /*
         *  Q.704/10.3.2 c): If local inhibiting is in
         *  effect, then the actions described 10.3.1 b),
         *  c), d) and e) take place.  If uninhibition is
         *  abandoned, step f) below is taken.
         */
        *f |= LK_FLAG_FORCEUNINHIBIT;
        *f |= LK_FLAG_UNINHIBITING;
        lk_send_uninhibit(lk);  /* send on concerned link */
        lk_timer_start(t12);
    }
    if ( *f & LK_FLAG_REM_INHIBITED ) {
        /*
         *  Q.704/10.3.2 d): If remote inhibiting is in
         *  effect, the signalling point X requests forced
         *  uninhibiting of the signalling link by sending a
         *  force uninhibit signalling link message to
         *  signalling point Y, which will then initiate
         *  uninhibiting in accordance with the description
         *  given in 10.3.1 b), c), d) and e).
         *
         *  The force uninhibit signalling link message is
         *  transmitted down the link to be uninhibited.
         */
        *f |= LK_FLAG_FORCEUNINHIBIT;
        lk_force_uninhibit(lk); /* LFU sent on concerned link */
        lk_timer_start(t13);
    }
}

static void
lk_t13_timeout(lk_t *lk) /* T13 timeout */
{
    register unsigned long *f = &lk->statem.flags;
    /*
     *  Q.704/10.3.2 e): If, for any reason, an unihibit
     *  signalling link message is not received in response
     *  to the force uninhibit message, a timer T13 expires.
     *  If this is the first expiry of T13 for this
     *  uninhibition attempt on this link, the procedure is
     *  restarted including inspection of the status of the
     *  inhibited link.  If the link is marked failed or
     *  blocked, or timer T13 has expired for the second
     *  time during inhibition of this link, management is
     *  informed and the uninhibition is abandoned.
     */
    if ( !(*f & LK_FLAG_FORCEUNINHIBIT) ) return;
    *f &= ~LK_FLAG_FORCEUNINHIBIT;
    if ( *f & LK_FLAG_FORCEUNINHIBIT2 ) {
        *f &= ~LK_FLAG_FORCEUNINHIBIT2;
        lk_route_uninhibit_abandoned(lk);
    }
    else
    {
        if ( *f & LK_MARK_UNUSABLE ) {
            lk_route_uninhibit_abandoned(lk);
            return;
        }
        *f |= LK_FLAG_FORCEUNINHIBIT | LK_FLAG_FORCEUNINHIBIT2;
        lk_send_force_uninhibit(lk); /* on concerned signalling link */
        lk_timer_start(t13);
    }
}

static inline void
lk_force_unihibit(lk_t *lk) /* LFU from adjacent L3 peer */
{
    register unsigned long *f = &lk->statem.flags;

    if ( !(*f & LK_FLAG_LOC_INHIBITED) ) {
        /*
         *  Q.704/10.4 c):  A force unihibit signalling link
         *  message concerning an unihibited link is
         *  answered with an uninhibit signalling link
         *  message without taking any further action.
         */
        lk_send_uninhibit(lk);
        return;
    }
    if ( *f & LK_MASK_UNUSABLE ) {
        /* nothing to do: lost communication with adjacent SP */
        return;
    }
    /*
     *  Q.704/10.3.1 b): If uninhibiting is possible,
     *  signalling point X sends uninhibit signalling link
     *  message to signalling point Y indicating that the
     *  link identified in the message should be
     *  uninhibited.
     */
    *f |= LK_FLAG_UNINHIBITING;
    lk_send_uninhibit(lk);
    lk_timer_start(t12);
}

/*
 *  10.6    Inhibit test procedure
 *
 *  When a signalling link becomes management inhibited, periodic tests are
 *  started to guard the inhibition status at each end of the link.
 *
 *  10.6.1  A local inhibit test is performed when timer T22 expires at
 *  signalling point X and the concerned link is marked locally inhibited.  In
 *  this case, a local inhibit test message is sent to the signalling point Y at
 *  the other end of the link, and timer T22 is restarted.
 *
 *  Reception of a local inhibit test message causes:
 *
 *  i)  no action, if the concerned link is marked remotely inhibited at the
 *      receiving signalling point Y; or
 *
 *  ii) the force uninhibit procedure to be involved at the receiving signalling
 *      point Y, if the concerned signalling link is not marked remotely
 *      inhibited at Y.  This procedure causes the locally inhibited status of
 *      the link at X to be cancelled.
 *
 *  If a timer T22 expires at the concerned link is not locally inhibited, no
 *  further action is taken.
 */

static void
lk_t22_timeout(lk_t *lk) /* T22 timeout */
{
    register unsigned long *f = &lk->statem.flags;
    /*
     * Q.704/10.6.1: A local inhibit test is performed when
     * timer T22 expires at signalling point X and the
     * concerned link is marked locally inhibited.  In this
     * case, a local inhibit test message is sent to the
     * signalling point Y at the other end of the link, and
     * timer T22 is restarted.
     */
    if ( !(*f & LK_FLAG_LOC_INHIBITED) )
        /*
         *  Q.704/10.6.1: ...  If a timer T22 expires at the
         *  concerned link is not locally inhibited, no
         *  further action is taken.
         */
        return;
    lk_send_local_inhibit_test(lk); /* FIXME: down what link? */
    lk_timer_start(t22);
}

static inline void
lk_local_inhibit_test(lk_t *lk) /* LLI from adjacent L3 peer */
{
    register unsigned long *f = &lk->statem.flags;
    /*
     *  Q.704/10.6.1: ..  Reception of a local inhibit test
     *  message causes:
     */
    if ( *f & LK_FLAG_REM_INHIBITED ) {
        /*
         *  Q.704/10.6.1 i)  no action, if the concerned
         *  link is marked remotely inhibited at the
         *  receiving signalling point Y; or
         */
        return;
    } else {
        /*
         *  Q.704/10.6.1 ii) the force uninhibit procedure
         *  to be involved at the receiving signalling point
         *  Y, if the concerned signalling link is not
         *  marked remotely inhibited at Y.  This procedure
         *  causes the locally inhibited status of the link
         *  at X to be cancelled.
         */
        if ( (*f & LK_FLAG_INACCESSIBLE) && (*f & LK_MASK_UNUSABLE) )
            return;
        *f |= LK_FLAG_FORCEUNINHIBIT;
        lk_send_force_uninhibit(lk); /* LFU sent on concerned link */
        lk_timer_start(t13);
    }
}

/*
 *  10.6.2  A remote inihibit test is performed when timer T23 expires at
 *  signalling point Y and the concerned link is marked remotely inhibited.  In
 *  this case a remote inhibit test message is sent to signalling point X at the
 *  other end of the link; and timer T23 is restarted.
 *
 *  Reception of a remote inhibit test message causes:
 *
 *  i)  no action, if teh concerned link is marked locally inhibited at the
 *      receiving signalling point X; or
 *
 *  ii) the uninhibit procedure to be invoked at the receiving signalling point
 *      X, if the concerned link is not marked locally inhibited at X.  This
 *      procedure causes the remotely inihibited status of the link at Y to be
 *      cancelled.
 *
 *  If a timer T23 expires and the concerned link is not remotely inhibited, no
 *  further action is taken.
 *
 */

static void
lk_t23_timeout(lk_t *lk) /* T23 timeout */
{
    register unsigned long *f = &lk->statem.flags;
    /*
     *  Q.704/10.6.2:  A remote inihibit test is performed
     *  when timer T23 expires at signalling point Y and the
     *  concerned link is marked remotely inhibited.  In
     *  this case a remote inhibit test message is sent to
     *  signalling point X at the other end of the link; and
     *  timer T23 is restarted.
     */
    if ( !(*f & LK_FLAG_REM_INHIBITED) )
        /*
         *  Q.704/10.6.2: ...  If a timer T23 expires and
         *  the concerned link is not remotely inhibited, no
         *  further action is taken.
         */
        return;
    lk_send_remote_inhibit_test(lk); /* FIXME: down what link? */
    lk_timer_start(t23);
}

static inline void
lk_remote_inhibit_test(lk_t *lk) /* LRI from adjacent L3 peer */
{
    register unsigned long *f = &lk->statem.flags;
    /*
     *  Q.704/10.6.2: ...  Reception of a remote inhibit
     *  test message causes:
     */
    if ( *f & LK_FLAG_REM_INHIBITED ) {
        /*
         *  Q.704/10.6.2 i) no action, if the concerned link
         *  is marked locally inhibited at the receiving
         *  signalling point X; or
         */
        return;
    } else {
        /*
         *  Q.704/10.6.2 ii) the uninhibit procedure to be
         *  invoked at the receiving signalling point X, if
         *  the concerned link is not marked locally
         *  inhibited at X.  This procedure causes the
         *  remotely inihibited status of the link at Y to
         *  be cancelled.
         */
        if ( (*f & LK_FLAG_INACCESSIBLE) && (*f & LK_MASK_UNUSABLE) )
            return;
        *f |= LK_FLAG_UNINHIBITING;
        lk_send_uninhibit(lk);
        lk_timer_start(t12);
    }
}

/*
 *  10      Management Inhibiting
 *
 *  10.1    General
 *
 *  Signalling link management inhibiting is requested by management when it
 *  becomes necessary, e.g. for maintenance or testing purposes (for example, if
 *  the link experiences too many changeovers and changebacks in a short time,
 *  or there is a significant link error rate), to make or keep a signalling
 *  link unavailable to User Part-generated signalling traffic.  Management
 *  inhibiting is a signalling traffic management action, and does not cause any
 *  link status changes at level 2.  A signalling link is marked "inhibited"
 *  under the management inhibiting procedure.  In particular, a signalling link
 *  that was active and in service prior to being inhibited will remain so, and
 *  will thus be able to transmit maintenance and test messages, for which the
 *  signalling link test message could be used (see clause 2/Q.707).
 *
 *  Inhibiting of a signalling link may be requested by management functions at
 *  either end of the link.  The request is granted, provided that the
 *  inhibiting action does not cause any previously accessible destination to
 *  become inaccessible at either end of the signalling link.  The request may
 *  also be refused under certain circumstances such as congestion.
 *
 *  A signalling link normally remains inhibited until uninhibiting is invoked
 *  in the signalling point at which inhibiting was initiated. Uninhibiting is
 *  initiated either at the request of a management function or by routing
 *  functions at either end of the signalling link when it is found that a
 *  destination has become inacessible for signalling traffic and the link sets
 *  associated with routes to that destination contains inhibited links.  Unless
 *  unavailable for other reasons, unihibiting causes the signalling link to
 *  enter the available state and changeback to be initiated.
 *
 *  Periodic tests are made on the inhibit status of inhibited links.  Such
 *  periodic tests should not add significantly to the traffic load on the
 *  signalling network, and remove the need for a signalling point to perform
 *  inhibit tests at signalling point restart.
 *
 *  If a test on the inhibit status of a link reveals discrepancies between the
 *  signalling points at each end of the link, the link is either uninhibited or
 *  force uninhibited as appropriate, to align the inhibit status at each end of
 *  the link.
 *
 *  10.2    Inhibiting initiation and actions
 *
 *  When at signalling point X a request is received from a management function
 *  to inhibit a signalling link to signalling point Y, the following action
 *  take place:
 *
 *  a)  A check if performed at signalling point X to determine whether, in the
 *      case of an available link, inhibiting will result in a destination
 *      becoming inaccessible, or in the case of an unavailable link, signalling
 *      point Y is inaccesible.  If either is the case, management is informed
 *      that the inhibiting request is denied.
 *
 *  b)  If inhibiting is permitted, signalling point X sends an inhibit message
 *      to signalling point Y indicating that it wishes to inhibit the
 *      signalling link identified in the message.
 *
 *  c)  Signalling point Y, on receiving the inhibit message from X, checks
 *      whether, in the case of an available link, inhibiting will result in a
 *      destination becoming inaccessible and, if so, an inhibit denied message
 *      is returned to signalling point X.  The latter then informs the
 *      management function which requested inhibiting that the request cannot
 *      be granted.
 *
 *  d)  If the signalling point Y finds that inhibiting of the concerned link is
 *      permissible, it sends an inhibit acknowledgement to signalling point X
 *      and marks the link remotely inhibited.
 *
 *      If the link concerned is currently carrying traffic, signalling point Y
 *      sends the inhibit acknowledgement via that link and diverts subsequent
 *      traffic for it, using the time controlled changeover procedure.  Y then
 *      starts inhibit test timer T23.
 *
 *  e)  On receiving an inhibit acknowledgement message, signalling point X
 *      marks the link locally inhibited and informs management that the link is
 *      inhibited.
 *
 *      If the link concerned is currently carrying traffic, signalling point X
 *      diverts subsequent traffic for that link, using the time-controlled
 *      changeover procedure.  X then starts inhibit test timer T22.
 *
 *  f)  When changeover has been completed, the link while inhibited, will be
 *      unavailable for the transfer of user-generated traffic but still permits
 *      the exchange of test messages.
 *
 *  g)  If, for any reason, the inhibit acknowledgement message is not received,
 *      a timer T14 expires and the procedure is restarted including inspection
 *      of the status of the destination of the inhibit message.  If the
 *      destination is not available, management is informed.
 *
 *  At most two consecutive automatic attempts may be made to inhibit a
 *  particular signalling link.
 *
 *  A signalling point may not transmit an inhibit message for a particular
 *  signalling link if it has already transmitted an uninhibit message for that
 *  link, and neither an acknowledgement for that uninhibit message has been
 *  received nor has the uninhibit procedure finally timed out.
 *
 *  10.3    Uninhibited initiation and actions
 *
 *  Signalling link uninhibiting is initiated at the signalling point which
 *  originally caused the link to be inhibited, upon receipt of an uninhibit or
 *  forced uninhibit request.
 *
 *  In a given signalling point, an uninhibit request may be initiated for a
 *  locally inhibited link by the management or signalling route control
 *  function, while a forced uninhibit request may be initiated for a remotely
 *  inhibited link by the signalling routing control function only.
 *
 *  Signalling routing control will initiate signalling link uninhibit if an
 *  inhibited link is found to be a member of a link set in a route to a
 *  destination which has become inaccessible.
 *
 *  If such signalling routing control uninhibiting were unsuccessful because of
 *  a failed or block inhibited link, and if that link later recovers or becomes
 *  unblocked with the destination still unavailable, uninhibiting is
 *  re-attempted.
 *
 *  A signalling point may not transmit an uninhibit message for a particular
 *  signalling link if it has already transmitted an inhibit message for that
 *  link, and neither an acknowledgement for that inhibit message has been
 *  received nor has the inhibit procedure finally timed out.
 *
 *  10.3.1  Management-initiated uninhibiting
 *
 *  Upon receipt of an uninhibiting request from the management function of
 *  signalling point X regarding an inhibited link to signalling point Y, the
 *  following actions take place:
 *
 *  a)  A check is performed at signlaling point X to determine whether an
 *      uninhibit message can be sent to signalling point Y, either over an
 *      available route, or if all routes to signalling point Y are unavailable,
 *      over the concerned inhibited link.  If all routes to signalling point Y
 *      are unavailable and the concerned inhibited link is marked failed or
 *      processor outage, management is informed that uninhibiting is not
 *      possible.
 *
 *  b)  If uninhibiting is possible, signalling point X sends uninhibit
 *      signalling link message to signalling point Y indicating that the link
 *      identified in the message should be uninhibited.
 *
 *  c)  Upon receipt of the uninhibit link message, signalling point Y returns
 *      an uninhibit acknowledgement message to signalling point X and cancels
 *      the remote inhibit indication.  If no local inhibited, failed or blocked
 *      condition exists on the link, it is put in the available state and
 *      changeback is initiated.
 *
 *  d)  On receipt of the uninhibit acknowledgement message, signalling point X
 *      cancels the local inhibit indication and informs management that the
 *      link has be uninhibited.  If no remote inhibited, failed or blocked
 *      condition exists on the link, it is put in the available state and
 *      changeback is initiated.
 *
 *  e)  If, for any reason, the uninhibit acknowledgement message is not
 *      received, a timer T12 expires.  If this is the first expirty of T12 for
 *      this uninhibition attempt on this link, the procedure is restarted
 *      including inspection of the status of the destination of the uninhibit
 *      message.  If the destination is not available, or T12 has expired for
 *      the second time during the uninhibition attempt on this link, management
 *      is informed, and the uninhibition is abandoned.
 *
 *  10.3.2  Signalling routing control initiated uninhibiting
 *
 *  Upon receipt of an uninhibit request for signalling routing control at
 *  signalling point X regarding an inhibited link to signalling point Y, the
 *  following actions take place:
 *
 *  a)  A check is performed at signalling point X to determine whether the
 *      concerned inhibited link is marked failed or blocked.  If it is, then
 *      signalling point X is unable to transmit an uninhibit message to
 *      signalling point Y, uninhibiting is therefore not possible, and the
 *      uninhibiting attempt is abandoned.
 *
 *  b)  If unihibiting is possible, a further check is performed by signalling
 *      point X to determine whether inhibiting initiated by X (local
 *      inhibiting) or inhibiting initiated by Y (remote inhibiting) is in
 *      effect.
 *
 *  c)  If local inhibiting is in effect, then the actions described 10.3.1 b),
 *      c), d) and e) take place.  If uninhibition is abandoned, step f) below
 *      is taken.
 *
 *  d)  If remote inhibiting is in effect, the signalling point X requests
 *      forced uninhibiting of the signalling link by sending a force uninhibit
 *      signalling link message to signalling point Y, which will then initiate
 *      uninhibiting in accordance with the description given in 10.3.1 b), c),
 *      d) and e).
 *
 *      The force uninhibit signalling link message is transmitted down the link
 *      to be uninhibited.
 *
 *  e)  If, for any reason, an unihibit signalling link message is not received
 *      in response to the force uninhibit message, a timer T13 expires.  If
 *      this is the first expiry of T13 for this uninhibition attempt on this
 *      link, the procedure is restarted including inspection of the status of
 *      the inhibited link.  If the link is marked failed or blocked, or timer
 *      T13 has expired for the second time during inhibition of this link,
 *      management is informed and the uninhibition is abandoned.
 *
 *  f)  If an attempt to uninhibit a signalling link is abandoned, signalling
 *      routing control attempts to uninhibit the next inhibited link to
 *      signaling point Y, starting from a) above.  The search continues until
 *      either a link is successfully uninhibited or all possible links to Y in
 *      the routing table have been exhausted, or the destination has become
 *      available for other reasons.
 *
 *  10.4    Receipt of unexpected management inhibition messages
 *
 *  a)  An inhibit signalling link message concerning an inhibited signalling
 *      link is answered with an inhibit acknowledgement message without taking
 *      any further action.
 *
 *  b)  An uninhibit signalling link message concerning an uninhibited
 *      signalling link is answered with an uninhibit acknowledgement message
 *      without taking any further action.
 *
 *  c)  A force unihibit signalling link message concerning an unihibited link
 *      is answered with an uninhibit signalling link message without taking any
 *      further action.
 *
 *  d)  If an inhibit acknowledgement message is received and no inhibit
 *      signalling link message is outstanding for the concerned link, no action
 *      is taken.
 *
 *  e)  If an uninhibit acknowledgement message is received and no unihibit
 *      signalling link message is outstanding for the concerned link, no action
 *      is taken.
 *
 *  10.5    Management inhibited link and processor recovery
 *
 *  a)  After a local processor recovery that involves the loss of inhibit
 *      status information, the signalling point will mark all links as
 *      unihibited, and message traffic will be restarted.
 *
 *  b)  If messages for Level 4 are received on an inhibited signalling link,
 *      the messages will be discriminated and distributed.
 *
 *  10.6    Inhibit test procedure
 *
 *  When a signalling link becomes management inhibited, periodic tests are
 *  started to guard the inhibition status at each end of the link.
 *
 *  10.6.1  A local inhibit test is performed when timer T22 expires at
 *  signalling point X and the concerned link is marked locally inhibited.  In
 *  this case, a local inhibit test message is sent to the signalling point Y at
 *  the other end of the link, and timer T22 is restarted.
 *
 *  Reception of a local inhibit test message causes:
 *
 *  i)  no action, if the concerned link is marked remotely inhibited at the
 *      receiving signalling point Y; or
 *
 *  ii) the force uninhibit procedure to be involved at the receiving signalling
 *      point Y, if the concerned signalling link is not marked remotely
 *      inhibited at Y.  This procedure causes the locally inhibited status of
 *      the link at X to be cancelled.
 *
 *  If a timer T22 expires at the concerned link is not locally inhibited, no
 *  further action is taken.
 *
 *  10.6.2  A remote inihibit test is performed when timer T23 expires at
 *  signalling point Y and the concerned link is marked remotely inhibited.  In
 *  this case a remote inhibit test message is sent to signalling point X at the
 *  other end of the link; and timer T23 is restarted.
 *
 *  Reception of a remote inhibit test message causes:
 *
 *  i)  no action, if teh concerned link is marked locally inhibited at the
 *      receiving signalling point X; or
 *
 *  ii) the uninhibit procedure to be invoked at the receiving signalling point
 *      X, if the concerned link is not marked locally inhibited at X.  This
 *      procedure causes the remotely inihibited status of the link at Y to be
 *      cancelled.
 *
 *  If a timer T23 expires and the concerned link is not remotely inhibited, no
 *  further action is taken.
 *
 */

/*
 * 
 *  5       Changeover
 *
 *
 *  5.1     General
 *
 *  5.1.1   The objective of the changeover procedures is to ensure that
 *  signalling traffice carried by the unavailable signalling link is diverted
 *  to the alternative signalling link(s) as quickly as possible while avoiding
 *  message loss, duplication or mis-sequencing.  For this purpose, in the
 *  normal case the changeover procedure includes buffer updating and retrieval,
 *  which are performed before reopening the alternative signalling link(s) to
 *  the diverted traffic.  Buffer updating consists of identifying all those
 *  messages in the retransmission buffer of the unavailable signalling link
 *  which have not been received by the far end.  This is done by means of a
 *  hand-shake procedure, based on changeover messages, performed between the
 *  two ends of the unavailable signalling link.  Retrieval consists of
 *  transferring the concerned messages to the transmission buffer(s) of the
 *  alternative link(s).
 *
 *  5.1.2   Changeover includes the procedures to be used in the case of
 *  unavailability (due to failure, blocking or inhibiting) of a signalling
 *  link, in order to diver the traffic pertaining to that signalling link to
 *  one or more alternative signalling links.
 *
 *  These signalling links can be carrying their own signalling traffic and this
 *  is not interrupted by the changeover procedure.
 *
 *  The different network configurations to which the changeover procedure may
 *  be applied are described in 5.2.
 *
 *  The criterion for intitiation of changeover, as well as the basic actions to
 *  be performed, are described in 5.3.
 *
 *  Procedures necessary to cater for equipment failure or other abnormal
 *  conditions are also provided.
 *
 *
 *  5.2     Network configuration for changeover
 *
 *  5.2.1   Signalling traffic diverted from an unavailable signalling link is
 *  routed by the concerned signalling point according to the rules specified in
 *  clause 4.  In summary, two alternative situtations may arise (either for the
 *  whole diverted traffic or for traffic relating to each particular
 *  destination):
 *
 *  i)      traffic is diverted to one or more signalling links of the same link
 *          set; or
 *
 *  ii)     traffic is diverted to one or more different link sets.
 *
 *  5.2.2   As a result of these arrangements, and of the message routing
 *  function described in clause 2, three different relationships between the
 *  new signalling link and the unavailable one can be identified, for each
 *  particular traffic flow.  These three basic cases may be summarized as
 *  follows:
 *
 *  a)      the new signalling link is parallel to the unavailable one (see
 *          Figure 9)
 *
 *  b)      the new signalling link belongs to a signalling route other than
 *          that to which the unavailable signalling link belongs, but this
 *          signalling route still passes through the signalling point at the
 *          far end of the unavailable signalling link (see Figure 10);
 *
 *  c)      the new signalling link belongs to a signalling route other than
 *          that to which the unavailable signalling link belongs, and this
 *          signalling route does not pass through the signalling point acting
 *          as signalling transfer point, at the far end of the signalling link
 *          (see Figure 11).
 *
 *  Only in the case of c) does a possibility of message mis-sequencing exist;
 *  therefore its use should take into account the overall service dependabilty
 *  requirements described in Recommendation Q.706.
 *
 *
 *  5.3     Changeover initiation and actions
 *
 *  5.3.1   Changeover is initiated at a signalling point when a signalling link
 *  is recognized as unavailable according to the criteria listed in 3.2.2.
 *
 *  The following actions are then performed:
 *
 *  a)      transmission and acceptance of message signal units on the concerned
 *          signalling link is terminated;
 *
 *  b)      transmission of link status signal units or fill in signal units, as
 *          described in 5.3/Q.703 take place;
 *
 *  c)      the alternative signalling link(s) are determined according to the
 *          rules specified in clause 4;
 *
 *  d)      a procedure to update the content of the retransmission buffer of
 *          the unavailable signalling link is performed as specified in 5.4
 *          below;
 *
 *  e)      signalling traffic is diverted to the alternative signalling link(s)
 *          as specified in 5.5 below.
 *
 *  In addition, if traffic towards a given destination is diverted to an
 *  alternative signalling link terminating in a signalling transfer point not
 *  currently used to carry traffic towards that destination, a
 *  transfer-prohibited procedure is performed as specified in 13.2.
 *
 *  5.3.2   In the case when there is no traffic to transfer from the
 *  unavailable signalling link action, only item b) of 5.3.1 is required.
 *
 *  5.3.3   If no alternative signalling link exists for signalling traffic
 *  towards one or more destinations, the concerned destination(s) are declared
 *  inaccessible and the following actions apply:
 *
 *  i)      the routing of the concerned signalling trafic is blocked and the
 *          concerned messages already stored in the transmission and
 *          retransmission buffers of the unavailable signalling link, as well
 *          as those received subsequently, are discarded;[3]
 *
 *          [3] The adequacy of this procedure to meet the acceptable
 *          dependability objectives in terms of loss of messages requires
 *          further study.
 *
 *  ii)     a command is sent to the User Part(s) (if any) in order to stop
 *          generating the concerned signalling traffic;
 *
 *  iii)    the transfer-prohibited procedure is performed, as specified in
 *          13.2;
 *
 *  iv)     the appropriate signalling link management procedures are performed,
 *          as specified in clause 12.
 *
 *  5.3.4   In some cases of failures or in some network configurations, the
 *  normal buffer updating and retrieval procedures described in 5.4 and 5.5
 *  cannot be accomplished.  In such cases, the emergency changeover procedure
 *  described in 5.6 apply.
 *
 *  Other procedures to cover possible abnormal cases appear in 5.7.
 *
 *
 *  5.4     Buffer updating procedure
 *
 *  5.4.1   When a decision to changeover is made, a changeover order is sent to
 *  the remote signalling point.  In the case that the changeover was initiated
 *  by the reception of a changeover order (see 5.2), a changeover
 *  acknowledgement is sent instead.
 *
 *  A changeover order is always acknowledged by a changeover acknowledgement,
 *  even when changeover has already been initiated in accordance with annother
 *  criterion.
 *
 *  No priority is given to the changeover order or changeover acknowledgement
 *  in relation to the normal traffic of the signalling link on which the
 *  message is sent.
 *
 *  5.4.2   The changeover order and changeover acknowledgement are signalling
 *  network management messages and contain the following information:
 *
 *  -       the label, indicating the destination and originating signalling
 *          points and the identity of the unavailable signalling link;
 *
 *  -       the changeover-order (or changeover-acknowledgement) signal; and
 *
 *  -       the forward sequence number of the last message signal unit accepted
 *          from the unavailable signalling link.
 *
 *  Formats and codes of the changeover order and the changeover acknowledgement
 *  appear in clause 15.
 *
 *  5.4.3   Upon reception of a changeover order or changeover acknowledgement,
 *  the retransmission buffer of the unavailable signalling link is updated
 *  (except as noted in 5.6), according to the information contained in the
 *  message.  The message signal units successive to that indicated by the
 *  message are those which have to be retransmitted on the alternative
 *  signalling link(s), according to the retrieval and diversion procedure.
 *
 *
 *  5.5     Retrieval and diversion of traffic
 *
 *  When the procedure to update the retransmission buffer content is completed,
 *  the following actions are performed:
 *
 *  -       the routing of the signalling traffic to be diverted is changed;
 *  
 *  -       the signal traffic already stored in the transmission buffers and
 *          retransmission buffer of the unavailable signalling link is sent
 *          directly towards the new signalling link(s), according to the
 *          modified routing.
 *
 *  The diverted signalling traffic will be sent towards the new signalling
 *  link(s) in such a way that the correct message sequence is maintained.  The
 *  diverted traffic has no priority in relation to normal traffic already
 *  conveyed on the signalling link(s).
 *
 *
 *  5.6     Emergency changeover procedures
 *
 *  5.6.1   Due to the failure in a signalling terminal, it may be impossible
 *  for the corresponding end of the faulty signalling link to determine the
 *  forward sequence number of the last message signal unit accepted over the
 *  unavailable link.  In this case, the concerned end accomplishes, if
 *  possible, the buffer updating procedures described in 5.4 but it make use of
 *  an emergency changeover order or and emergency changeover acknowledgement
 *  instead of the corresponding normal message; these emergency messages, the
 *  format of which appears in clause 15, do not contain the forward sequence
 *  number of the last accepted message signal unit.  Furthermore, the
 *  signalling link is taken out of service, i.e. the concerned end initiates,
 *  if possible, the sending of out-of-service link status signal units on the
 *  unavailable link (see 5.3/Q.703).
 *
 *  When the other end of the unavailable signalling link receives the emergency
 *  changeover order or acknowledgement, it accomplishes the changeover
 *  procedures described in 5.4 and 5.5, the only difference being that it does
 *  not perform either buffer updating or retrieval.  Instead, it directly
 *  starts sending the signalling traffic not yet transmitted on the unavailable
 *  link on the alternative signalling link(s).
 *
 *  The use of normal or emergency changeover messages depends on the local
 *  conditions of the sending signalling point only, in particular:
 *
 *  -       an emergency changeover order is acknowledged by a changeover
 *          acknowledgement if the local conditions are normal; and
 *
 *  -       a changeover order is acknowledged by an emergency changeover
 *          acknowledgement if there are local fault conditions.
 *
 *  5.6.2   Time-controlled changeover is initiated when the exchange of
 *  changeover messages is not possible or not desirable, i.e. if any (or
 *  several) of the following cases apply:
 *
 *  i)      No signalling path exists between the two ends of the unavailable
 *          link; so that the exchange of changeover messages is impossible.
 *
 *  ii)     Processor outage indication is received on a link.  In this case, if
 *          the remote processor outage condition is only transitory, sending of
 *          a changeover order could result in failure of the link.
 *
 *  iii)    A signalling link currently carrying traffic has been marked
 *          (locally or remotely) inhibited.  In this case, time controlled
 *          changeover is used to divert traffic for the inhibited link without
 *          causing the link to fail.
 *
 *  When the concerned signalling point decides to initiate changeover in such
 *  circumstances, after the expiry of timer T1 (see 16.8), it starts signalling
 *  traffic not yet transmitted on the unavailable signalling link on the
 *  alternative link(s); the purpose of witholding traffic for time T1 (see
 *  16.8) is to reduce the probability of message mis-sequencing.
 *
 *  An example of such a case appears in Annex A/Q.705.
 *
 *  In the abnormal case when the concerned signalling point is not aware of the
 *  situation, it will start the normal changeover procedure and send a
 *  changeover order; in this case it will receive no changeover message in
 *  response and the procedure will be completed as indicated in 5.7.2.
 *  Possible reception of a transfer-prohibited message (sent by an involved
 *  signalling transfer point on reception of the changeover order, see 13.2)
 *  will not affect changeover procedures.
 *
 *  If time-controlled changeover has been initiated according to case ii) above
 *  and if a changeover order is received from the remote and during the Time
 *  T1, it is advantageous to switch to the normal changeover procedure
 *  including retrieval because unecessary message loss or sending of old
 *  messages is avoided in a simple way.  The ability to perform this switch is
 *  considered to be implementation dependent.  A changeover acknowledgement
 *  however, must be returned in any case in order to assure the normal
 *  completion of the changeover procedure at the remote end.  If a changeover
 *  order is received after timer T1 has expired time-controlled changeover is
 *  completed (if not yet done) and an emergency changeover acknowledgement is
 *  sent to the remote end.
 *
 *  In the case that processor outage is of long-term, the remote side completes
 *  the time-controlled changeover procedure.  In order to avoid sending out old
 *  messages (see clause 8/Q.703) the level 2 buffers on both sides of the link
 *  should be flushed immediately, when the local/remote processor outage state
 *  terminates.  How the flushing is performed is implementation dependent.  The
 *  decision whether processor outage is of long-term is a local thing.  At the
 *  remote side long-term processor outage occurs when the time-controlled
 *  changeover timer T1 expires.  At the local side an equivalent timer is used
 *  in quite the same way.
 *
 *  5.6.3   Due to failures, it may be impossible for a signalling point to
 *  perform retrieval even if it has received the retrieval information from the
 *  far end of the unavailable signalling link.  In this case, it starts sending
 *  new traffic on reception of the changeover message (or on time-out expiry,
 *  see 5.6.2 and 5.7.2); no further actions in addition to the other normal
 *  changeover procedures are performed.
 *
 *
 *  5.7     Procedures in abnormal conditions
 *
 *  5.7.1   The procedures described in this subclause allow the completion of
 *  the changeover procedures in abnormal cases other than those described in
 *  5.6.
 *
 *  5.7.2   If no changeover message in response to a changeover order is
 *  received within a timer T2 (see 16.8), new traffic is started on the
 *  alternative signalling link(s).
 *
 *  5.7.3   If a changeover order or acknowledgement containing an unreasonable
 *  value of the forward sequence number is received, no buffer updating or
 *  retrieval is performed, and new traffic is started on the alternative
 *  signalling link(s).
 *
 *  5.7.4   If a changeover acknowledgement is received without having
 *  previously sent a changeover order, no action is taken.
 *
 *  5.7.5   If a changeover order is received relating to a particular
 *  signalling link after the completion of changeover from that signalling
 *  link, an emergency changeover acknowledgement is sent in repsonse, without
 *  any further action.
 *
 */

/*
 *
 *  6       Changeback
 *
 *
 *  6.1     General
 *
 *  6.1.1   The objective of the changeback procedure is to ensure that
 *  signalling traffic is diverted from the alternative signalling link(s) to
 *  the signalling link made available as quickly as possible, while avoiding
 *  message loss, duplication or mis-sequencing.  For this purpose (in the
 *  normal case), changeback includes a procedure to control the message
 *  sequence.
 *
 *  6.1.2   Changeback includes the basic procedures to be used to perform the
 *  opposite action to changeover, i.e. to divert traffic from the alternative
 *  signalling link(s) to a signalling link which has become available (i.e. it
 *  was uninhibited, restored or unblocked).  The characteristics of the
 *  alternative signalling link(s) from which changeback can be made are
 *  described in 5.2.  In all the cases mentioned in 5.2, the alternative
 *  signalling links can be carrying their own signalling traffic and this is
 *  not interrupted by the changeback procedures.
 *
 *  Procedures necessary to cater for particular network configuration or other
 *  abnormal conditions are also provided.
 *
 *  NOTE - The term "alternative signalling link(s)" refers to signalling
 *  link(s) terminating in the signalling point at which a changeback is
 *  initiated (see also clause 4).
 *
 *
 *  6.2     Changeback initiation and action
 *
 *  6.2.1   Changeback is initiated at a signalling point when a signalling link
 *  is restored, unblocked or uninhibited, and therefore it becomes once again
 *  available, according to the criteria listed in 3.2.3 and 3.2.7.  The
 *  following actions are then performed:
 *
 *  a)      the altnerative signalling link(s) to which traffic normally carried
 *          by the signalling link made available was previously diverted (e.g.
 *          on occurence of a changeover), are determined.  To this set are
 *          added, if applicable, other links determined as defined in 4.2.2;
 *
 *  b)      signalling traffic is diverted (if appropriate, according to the
 *          criteria specified in clause 4) to the concerned signalling link by
 *          means of the sequence control procedure specified in 6.3; traffic
 *          diversion can be performed at the discretion of the signalling point
 *          initiating the changeback, as follows:
 *
 *          i)   individually for each traffic flow (i.e. on destination basis);
 *
 *          ii)  individually for each alternative signalling link (i.e. for all
 *               destinations previously diverted on that alternative signalling
 *               link);
 *
 *          iii) at the same timer for a number of, or for all the alternative
 *               signalling links.
 *          
 *  On occurence of changeback, it may happen that the traffic towards a given
 *  destination is no longer routed via a given adjacent signalling transfer
 *  point, towards which a transfer-prohibited procedure was previously
 *  performed on occurence of changeover (see 5.3.1); in this case a
 *  transfer-allowed procedure is performed as specified in 13.3.
 *
 *  In addition, if traffic towards a given destination is diverted to an
 *  alternative signalling link terminating in a signalling transfer point not
 *  currently used to carry traffic toward that destination, a
 *  transfer-prohibited procedure is performed as specified in 13.2.
 *
 *  6.2.2   In the case when there is no traffic to transfer to the signalling
 *  link made available, none of the previous actions are performed.
 *
 *  6.2.3   In the case that the signalling link made available can be used to
 *  carry signalling traffic towards a non-adjacent destination which was
 *  previously declared inaccessible, the following actions apply:
 *
 *  i)      the routing of the concerned signalling traffic is unblocked and
 *          transmission of the concerned messages (if any) is started on the
 *          link made available;
 *
 *  ii)     an indication is sent to the User Part(s) (if any) to restart the
 *          concerned signalling traffic;
 *
 *  iii)    the transfer-allowed procedure is performed, as specified in 13.3.
 *          However, in national networks, when the recovered link is not on the
 *          normal route for that destination, the transfer-restricted[1]
 *          procedure may be performed as specified in 13.4.
 *
 *  iv)     the transfer-prohibited procedure is performed as specified in
 *          13.2.2 i).
 *
 *  6.2.4   In the case that the signalling link made available will be the
 *  first link to be used on the normal route towards a destination previously
 *  declared restricted, the status of the route is changed to available and the
 *  transfer-allowed procedure is performed as specified in 13.3.
 *
 *  6.2.5   If the signalling point at the far end of the link made available
 *  currently is inacessible, from the signalling point initiating changeback
 *  (see clause 9 on MTP Restart), the sequence control procedure specified in
 *  6.3 (which requires communication between the two concerned signalling
 *  points) does not apply; instead, the time-controlled diversion specified in
 *  6.4 is performed.  This is made also when the concerned signalling points
 *  are accessiable, but there is no signalling route to it using the same
 *  outgoing signalling link(s) (or one of the same signalling link(s) from
 *  which traffic will be diverted).
 *
 *  The time-controlled diversion procedure may also be used for the changeback
 *  between different link sets instead of the sequence control procedure in
 *  order to avoid possible message mis-sequencing (see Note) or problems with
 *  multiple parallel changebacks.
 *
 *  NOTE - THe sequence control procedures can only guarantee correct sequencing
 *  of MSUs in all cases if the alternative link terminates in the same
 *  signalling point (i.e. the destination of the changeback declaration) as the
 *  newly available one.
 *
 *
 *  6.3     Sequence control procedure
 *
 *  6.3.1   When a decision is made at a given signalling point to divert a
 *  given traffic flow (towards one or more destinations) from an alternative
 *  signalling link to the signalling link made available, the following actions
 *  are performed if possible (see 6.4):
 *
 *  i)      transmission of the concerned traffic on the alternative signalling
 *          link is stopped; such traffic is stored in a changeback buffer,
 *
 *  ii)     a changeback declaration is sent to the remote signalling point of
 *          the signalling link made available via the concerned alternative
 *          signalling link; this message indicates that no more message signal
 *          units relating to the traffic being diverted to the link made
 *          available will be sent on the alternative signalling link.
 *
 *  6.3.2   The concerned signalling point will restart diverted traffic over
 *  the signalling link made available when it receives a changeback
 *  acknowledgement from the far signalling point of the link made available;
 *  this message indicates that all signal messages relating to the concerned
 *  traffic flow and routed to the remote signalling point via the alternative
 *  signalling link have been received.  The remote signalling point will send
 *  the changeback acknowledgement to the signalling point initiating changeback
 *  in response to the changeback declaration; any available signalling route
 *  between the two signalling points can be used to carry the changeback
 *  acknowledgement.
 *
 *  6.3.3   The changeback declaration and changeback acknowledgement are
 *  signalling network management messages and contain:
 *
 *  -       the label indicating the destination and originating signalling
 *          points, and the identity of the signalling link to which traffic
 *          will be diverted;
 *
 *  -       the changeback-declaration (or changeback-acknowledgement) signal;
 *          and
 *
 *  -       the changeback code.
 *
 *  Formats and codes fo the changeback-declaration and changeback-
 *  acknowledgement appear in clause 15.
 *
 *  6.3.4   A particular configuration of the changeback code is autonomously
 *  assigned to the changeback declaration by the signalling point initiating
 *  changeback; the same configuration is included in the changeback
 *  acknowledgement by the acknowledging signalling point.  This allows
 *  discrimination between different changeback declarations and
 *  acknowledgements when more than on sequence control procedures are initiated
 *  in parallel, as follows.
 *
 *  6.3.5   In the case that a signalling point intends to initiate changeback
 *  in parallel from more than one alternative signalling link, a sequence
 *  control procedure is accomplished for each involved signalling link, and a
 *  changeback-declaration is sent on each of them; each changeback declaration
 *  is assigned a different configuration of the changeback code.  Stopped
 *  traffic is stored in one or more changeback buffers (in the latter case, a
 *  changeback buffer is provided for each alternative signalling link).  When
 *  the changeback-acknowledgement relating to that alternative signalling link
 *  is received, traffic being diverted from a given alternative signalling link
 *  can be restarted on the signalling link made available, starting with the
 *  content of the changeback buffer; discrimination between the different
 *  changeback-acknowledgements is made by the changeback code configuration,
 *  which is the same as that sent in the changeback-declaration.
 *
 *  This procedure allows either reopening the recovered signalling link to
 *  traffic in a selective manner (provided that different changeback buffers
 *  are used) as soon as each changeback-acknowledgement is received, or only
 *  when all the changeback-acknowledgements have been received.
 *
 *
 *  6.4     Time-controlled diversion procedure
 *
 *  6.4.1   The time-controlled diversion procedure is used at the end of the
 *  MTP restart procedure (see clause 9) when an adjacent signalling point
 *  becomes available, as well as for the reasons given in 6.2.5.  An example of
 *  such use appears in Figure 12.
 *
 *  In this example, on failure of signalling link AB, traffic towards the
 *  destination D was directed to signalling link AC.  When AB becomes
 *  available, the point A considers itself as the neighbor of a point which
 *  restarts and applies the MTP restart procedure (see clause 9).
 *
 *  6.4.2   When changeback is initiated after the MTP restart procedure, the
 *  adjacent signalling point of the point whose MTP is restarting stops traffic
 *  to be directed from alternative signalling link(s) for a time T3, after
 *  which it starts traffic on the signalling link(s) made available.  The time
 *  delay minimizes the probabilty of out-of-sequence delivery to the
 *  destination point(s).
 *
 *
 *  6.5     Procedures in abnormal conditions
 *
 *  6.5.1   If a changeback-acknowledgement is received by a signalling point
 *  which has not previously sent a changeback-declaration, no action is taken.
 *
 *  6.5.2   If a changeback-declaration is received after the completion of the
 *  changeback procedure, a changeback-acknowledgement is sent in repsonse,
 *  without taking any further action.  This corresponds to the normal action
 *  described in 6.3.2 above.
 *
 *  6.5.3   If no changeback-acknowledgement is received in response to a
 *  changeback declaration within a time T4 (see 16.8), the
 *  changeback-declaration is repeated and a new timer T5 (see 16.8) is started.
 *  If no changeback-acknowledgement is received before the expiry of T5, the
 *  maintenance functions are alerted and traffic on the link made available is
 *  started.  The changeback code contianed in the changeback-acknowlegement
 *  message makes it possible to determine, in the case of parallel changebacks
 *  from more than one reserve path, which changeback-declaration is
 *  unacknowledged and has therefore to be repeated.
 *
 */
#endif  __LK_SM_H__

